/****************************************************************************
 *
 * Copyright (c) 1992-1996, John Forkosh Associates.  All rights reserved.
 * --------------------------------------------------------------------------
 *
 * Functions:	--- TexPerfect file control block constructor/destructor ---
 *		tpopen(wpfile,tpfile,msglevel) Opens WordPerfect and TeX
 *			files, validates 16-byte prefix, mallocs buffers.
 *		tpclose(tpfcb) Closes files and frees memory buffers.
 *		--- WordPerfect document file character input functions ---
 *		wpgetc(tpfcb,inbuff) Reads next char from file.
 *		wpregetc(tpfcb,inbuff) Returns most recently read character.
 *		wpungetc(tpfcb,inbuff,wpbyte) Pushes wpbyte onto unget buffer.
 *		wpungets(tpfcb,inbuff,wpstring) Pushes string onto unget buff.
 *		wpread(tpfcb,inbuff,buffer,itemsz,nitems) Performs an fread().
 *		--- TexPerfect file character output functions ---
 *		tpputc(tpfcb,tpbyte) Writes tpbyte to file (or buffer).
 *		tpoutc(tpfcb,outbuff,tpbyte) Writes tpbyte to file or outbuff.
 *		tpputs(tpfcb,outbuff,tpstring,nchars) Writes tpstring.
 *		--- Ancillary functions ---
 *		texcommand(tpfcb,icommand,istring)Gets string(s) for icommand.
 * 		setmode(tpfcb,mode) Set TeX mode (PARMODE,MATHMODE,etc).
 *		setwpchartbl() inits and returns (char ***) addr of wpchartbl.
 *		tpmalloc(nbytes) mallocs a MEMBUFF structure.
 *		tpfree(membuff) frees membuff structure.
 *
 * Notes:     o	See individual function headers.
 *
 * Source:	TPTOOLS.C
 *
 * --------------------------------------------------------------------------
 * Revision History:
 * 12/19/92	J.Forkosh	Installation.
 * 08/08/96	J.Forkosh	Modified and streamlined for copyleft release
 *
 ***************************************************************************/

/* --------------------------------------------------------------------------
Headers and Static Data
-------------------------------------------------------------------------- */
/* --- standard headers --- */
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
/* --- application headers --- */
#include "texperf.h"
#include "wpchrset.h"
#include "texcmds.h"

/****************************************************************************
 * Function:	tpopen ( wpfile, tpfile, msglevel )
 * Purpose:	Opens a WordPerfect document file and validates its contents,
 *		malloc()'s a TexPerf file control block if the file is ok,
 *		and opens a TexPerfect file for output.
 * --------------------------------------------------------------------------
 * Arguments:	wpfile (I)	address of char containing filename string
 *				(including any required path information)
 *				of WordPerfect document file to be opened.
 *		tpfile (I)	address of char containing filename string
 *				(including any required path information)
 *				of TexPerfect output file to be opened.
 *		msglevel (I)	int containing level of output requested.
 * Returns:	(TPFCB *)	pointer to TexPerfect file control block,
 *				or NULL for any error
 * --------------------------------------------------------------------------
 * Notes:     o
 ***************************************************************************/
/* --- entry point --- */
TPFCB	*tpopen ( wpfile, tpfile, msglevel )
char	*wpfile;
char	*tpfile;
int	msglevel;
{
/* --------------------------------------------------------------------------
Allocations and Declarations
-------------------------------------------------------------------------- */
TPFCB	*tpfcb=(TPFCB *)NULL;		/* fcb returned to caller */
FILE	*wfp=(FILE *)NULL,		/* pointer to WordPerfect file */
	*tfp=(FILE *)NULL,		/* pointer to TexPerfect file */
	*fopen();			/* prototype */
WPFILEPREF wpfilepref;			/* 16-byte WordPerfect file prefix */
MEMBUFF	**membuff=NULL, **buffptrs[16];	/* membuffs that are initialized */
int	nmembuff = 0;			/* #membuffs initialized */
int	okstatus = 0;			/* init to signal failure */

/* --------------------------------------------------------------------------
Validate input filename arguments, and open the WordPerfect file
-------------------------------------------------------------------------- */
buffptrs[0] = (MEMBUFF **)NULL;		/*terminate array in case of error*/
/* --- validate filename arguments passed by caller --- */
if ( wpfile == (char *)NULL		/* no wp filename argument passed */
||   tpfile == (char *)NULL )		/* no tp filename argument passed */
  goto end_of_job;			/* return with NULL fcb for error */
if ( msglevel >= 9 )
  printf("tpopen> TexPerfect initializing WordPerfect file=\"%s\"\n",wpfile);
if ( *wpfile == '\000'			/* null wp filename string passed */
||   *tpfile == '\000' )		/* null tp filename string passed */
  goto end_of_job;			/* return with NULL fcb for error */
/* --- open the WordPerfect file for binary read --- */
if ( (wfp = fopen(wpfile,"rb") )	/*open WordPerf file for binary read*/
==   (FILE *)NULL )			/* failed to open requested file */
  { printf("tp> can't find input file %s\n",wpfile); /* report problem */
    goto end_of_job; }			/* return with NULL fcb for error */

/* --------------------------------------------------------------------------
Read and validate WordPerfect document file prefix
-------------------------------------------------------------------------- */
/* --- read 16-byte prefix --- */
if ( fread((void *)(&wpfilepref),1,16,wfp) /* read 16 bytes into wpfilepref */
!=   16 )				/* failed to read prefix */
  { printf("tp> unable to read input file %s\n",wpfile); /* report problem */
    goto end_of_job; }			/* return with NULL fcb for error */
/* --- validate --- */
if ( memcmp(wpfilepref.fileid,"\377WPC",4) != 0 /*check for "\377WPC" tag*/
||   (int)(wpfilepref.prodtype) != 1	/* not a WordPerfect file */
||   (int)(wpfilepref.filetype) != 10	/* not a document file */
/*||   (int)(wpfilepref.majorver) != 0*/	/* not WordPerfect 5.x */
/*||   (int)(wpfilepref.minorver) != 1*/	/* not 5.1 */
||   (int)(wpfilepref.encryptkey) != 0	/* file is encrypted */
||   (int)(wpfilepref.reserved) != 0 )	/* reserved word is corrupted */
  { printf("tp> input file %s is not a WordPerfect 5.1 file\n",wpfile);
    goto end_of_job; }			/* return with NULL fcb for error */

/* --------------------------------------------------------------------------
Open the TexPerfect file for ascii or binary output (if caller passed tp name)
-------------------------------------------------------------------------- */
/* --- open the TexPrefect file for ascii write --- */
tfp = fopen(tpfile,"wb");		/*open TexPerf file for binary write*/
if ( tfp == (FILE *)NULL )		/* failed to open requested file */
  { printf("tp> can't create output file %s\n",tpfile); /*report problem*/
    goto end_of_job; }			/* return with NULL fcb for error */

/* --------------------------------------------------------------------------
Allocate and initialize file control block and memory buffers
-------------------------------------------------------------------------- */
/* --- allocate and initialize memory for the file control block --- */
if ( (tpfcb = (TPFCB *)(malloc(sizeof(TPFCB)))) /* malloc the fcb */
==   (TPFCB *)NULL )			/* failed to malloc */
  goto end_of_job;			/* return with NULL fcb for error */
memset((void *)tpfcb,0,sizeof(TPFCB));	/* initialize tpfcb to binary 0's */
strcpy(tpfcb->ident,"tpfcb");		/* initialize identification field */
tpfcb->msglevel = 1;			/* default message level */
tpfcb->textype = TEXTYPE;		/* default translation requested */
tpfcb->texmode = PARMODE;		/* default paragraph mode */
/*tpfcb->tpvars = (VARDEF *)NULL;*/	/* address of input variable struct */
tpfcb->msglevel = msglevel;		/* set message level */
/* --- set buffers to be initialized --- */
buffptrs[nmembuff++] = &(tpfcb->wpdoc.filebuff); /* wpgetc() input buffer */
buffptrs[nmembuff++] = &(tpfcb->tpfile.filebuff); /* output buffer */
buffptrs[nmembuff] = (MEMBUFF **)NULL;	/* trailer pointer */
/* --- allocate and initialize buffers --- */
for (nmembuff=0; (membuff=buffptrs[nmembuff])!=(MEMBUFF **)NULL; nmembuff++)
  if ( (*membuff = tpmalloc(MEMBUFFSIZE)) /* malloc and initialize */
  ==   (MEMBUFF *)NULL ) goto end_of_job; /* error, failed to malloc */

/* --------------------------------------------------------------------------
Initialize fields in file control block
-------------------------------------------------------------------------- */
/* --- initialize WordPerfect file and memory buffers --- */
strcpy(tpfcb->wpdoc.filename,wpfile);	/* filename of WordPerfect document */
strcpy(tpfcb->wpdoc.accessmode,"rb");	/* WordPerfect document access mode */
tpfcb->wpdoc.fp = wfp;			/* pointer to file */
/* --- initialize WordPerfect tables --- */
tpfcb->wpcodetblsz = wpcodetblsz;	/* #entries in wpcodetbl[] */
tpfcb->wpcodetbl = wpcodetbl;		/* char indexes its code type */
tpfcb->wpchartbl = wpchartbl;		/* extended character sets */
tpfcb->argstr = (char *)NULL;		/* argument passed by caller */
/* --- initialize WordPerfect file prefix --- */
memcpy(	(void *)(&(tpfcb->wpfilepref)),	/* file prefix in structure */
	(void *)(&wpfilepref), 16);	/* local file prefix */
tpfcb->wpdocoffset = (long)(wpfilepref.docoffset); /* doc offset from pref */
tpfcb->wpdocsize = tpfcb->wpdocoffset;	/* start with prefix byte count */
/* --- initialize TexPerfect file and memory buffers --- */
strcpy(tpfcb->tpfile.filename,tpfile);	/* filename of TexPerfect output */
strcpy(tpfcb->tpfile.accessmode,"wb");	/* TexPerfect access mode */
tpfcb->tpfile.fp = tfp;			/* pointer to file */
tpfcb->tokenlen = 0;			/* #chars in current token */
tpfcb->isheader = 0;			/* true after header written */
tpfcb->isdblspace = 1;			/* true for extra CR after HARDCR */
tpfcb->nhardcr = 0;			/* # consecutive hard cr's */

/* --------------------------------------------------------------------------
Back to caller with file control block (or NULL for error) at end-of-job.
-------------------------------------------------------------------------- */
okstatus = 1;				/*normal completion (no jump to eoj)*/
end_of_job:
  if ( !okstatus )			/* tpopen() failed */
    {
    if ( wfp != (FILE *)NULL ) fclose(wfp); /* close WordPerfect file */
    if ( tfp != (FILE *)NULL ) fclose(tfp); /* close TexPerfect file */
    if ( tpfcb != (TPFCB *)NULL ) free((void *)tpfcb); /* free fcb */
    for(nmembuff=0;(membuff=buffptrs[nmembuff])!=(MEMBUFF **)NULL;nmembuff++)
      if( membuff != (MEMBUFF **)NULL ) tpfree(membuff); /* free membuffs */
    tpfcb = (TPFCB *)NULL;		/* error returned to caller */
    if ( msglevel >= 1 )		/* report error */
      printf("tpopen> can't continue, returning to DOS\n");
    } /* --- end-of-if(!okstatus) --- */
  if ( msglevel >= 9 )
    printf("tpopen> TeXPerfect initialization completed\n");
  return ( tpfcb );			/* fcb pointer (or NULL) to caller */
} /* --- end-of-function tpopen() --- */


/****************************************************************************
 * Function:	tpclose ( tpfcb )
 * Purpose:	Closes files and frees buffers associated with
 *		file control block.
 * --------------------------------------------------------------------------
 * Arguments:	tpfcb (I)	address of TPFCB file control block
 *				containing file pointers to be closed
 *				and memory pointers to be freed.
 * Returns:	(int)		1 if successful
 *				or 0 for any error
 * --------------------------------------------------------------------------
 * Notes:     o
 ***************************************************************************/
/* --- entry point --- */
int	tpclose ( tpfcb )
TPFCB	*tpfcb;
{
/* --------------------------------------------------------------------------
Allocations and Declarations
-------------------------------------------------------------------------- */
int	okstatus = 0;			/* init to signal failure */
FILE	*fp = (FILE *)NULL;		/* file pointer */
void	*buff = (void *)NULL;		/* buffer pointer */

/* --------------------------------------------------------------------------
Validate file control block argument passed by caller.
-------------------------------------------------------------------------- */
/* --- validate file control block argument passed by caller --- */
if ( tpfcb == (TPFCB *)NULL )		/* no fcb argument passed */
  goto end_of_job;			/* return with status=0 for error */
if ( memcmp(tpfcb->ident,"tpfcb",5) != 0 ) /* not a file control block */
  goto end_of_job;			/* return with status=0 for error */

/* --------------------------------------------------------------------------
Close files, free buffers, and free the tpfcb itself.
-------------------------------------------------------------------------- */
/* --- close files --- */
if ( (fp = tpfcb->wpdoc.fp)		/* pointer to WordPerfect document */
!=   (FILE *)NULL ) fclose(fp);		/* close WordPerfect file */
if ( (fp = tpfcb->tpfile.fp)		/* pointer to TexPerfect file */
!=   (FILE *)NULL ) fclose(fp);		/* close TexPerfect file */
/* --- free buffers --- */
tpfree(tpfcb->wpdoc.filebuff);		/* WordPerfect input memory buffer */
tpfree(tpfcb->tpfile.filebuff);		/* TeXPerfect output memory buffer */
/* --- free tpfcb --- */
free((void *)tpfcb);			/* free the file control block */

/* --------------------------------------------------------------------------
Back to caller with status of tpclose() request
-------------------------------------------------------------------------- */
okstatus = 1;				/*normal completion (no jump to eoj)*/
end_of_job:
  return ( okstatus );			/* 1=success, 0=error */
} /* --- end-of-function tpclose() --- */


/****************************************************************************
 * Function:	wpgetc ( tpfcb, inbuff )
 * Purpose:	Reads next char from file or from unget buffer,
 *		or from input stream inbuff.
 * --------------------------------------------------------------------------
 * Arguments:	tpfcb (I)	address of file control block
 *				containing file pointer of WP document file.
 *		inbuff (I)	address of MEMBUFF normally containing NULL
 *				to read from the WordPerfect file,
 *				or containing an input buffer to be processed.
 * Returns:	(int)		character from file (or unget buffer),
 *				or EOF for any error
 * --------------------------------------------------------------------------
 * Notes:     o
 ***************************************************************************/
/* --- entry point --- */
int	wpgetc ( tpfcb, inbuff )
TPFCB	*tpfcb;
MEMBUFF	*inbuff;
{
/* --------------------------------------------------------------------------
Allocations and Declarations
-------------------------------------------------------------------------- */
int	wpbyte = EOF;			/* init to signal failure */
FILE	*fp = tpfcb->wpdoc.fp;		/* file pointer */
unsigned char *buff = tpfcb->wpdoc.filebuff->buffer; /* buffer pointer */
int	buffsize = tpfcb->wpdoc.filebuff->buffsize, /* allocated buffer size */
	buffhead = tpfcb->wpdoc.filebuff->buffhead, /* buffer head offset */
	bufftail = tpfcb->wpdoc.filebuff->bufftail; /* buffer tail offset */

/* --------------------------------------------------------------------------
Check whether special buffer provided
-------------------------------------------------------------------------- */
if ( inbuff != (MEMBUFF *)NULL )	/* user provided input buffer */
  {
  buffhead = inbuff->buffhead;		/* buffer head offset */
  bufftail = inbuff->bufftail;		/* buffer tail offset */
  if ( buffhead < 0			/* EOF for illegal index or */
  ||   buffhead >= bufftail ) goto end_of_job; /* at end-of-buffer */
  wpbyte = (inbuff->buffer)[buffhead];	/* extract byte */
  inbuff->buffchar = wpbyte;		/* store current char in buffer */
  inbuff->buffhead += 1;		/* bump to next byte */
  goto end_of_job;			/* return byte to caller */
  } /* --- end-of-if(inbuff!=NULL) --- */

/* --------------------------------------------------------------------------
Take char from unget buffer if available, or read directly from file
-------------------------------------------------------------------------- */
if ( bufftail != buffhead ) /* --- have chars in unget buffer --- */
  { wpbyte = (int)(buff[buffhead]);	/* retrieve leading char */
    buffhead++;				/* increment buffer head */
    if ( buffhead >= buffsize )		/* passed end-of-buffer */
      buffhead = 0;			/* so wrap to beginning */
    tpfcb->wpdoc.filebuff->buffhead = buffhead; } /* reset buffer head */
else /* --- nothing in unget buffer --- */
  { wpbyte = fgetc(fp);			/* just read next char from file */
    tpfcb->wpdocsize += (1L); }		/* and count one more byte read */
tpfcb->wpdoc.filebuff->buffchar = wpbyte; /* store current char in buffer */
end_of_job: return ( wpbyte );		/* char back to caller */
} /* --- end-of-function wpgetc() --- */


/****************************************************************************
 * Function:	wpregetc ( tpfcb, inbuff )
 * Purpose:	Rereads most recent char returned by wpgetc().
 * --------------------------------------------------------------------------
 * Arguments:	tpfcb (I)	address of file control block
 *				containing file pointer of WP document file.
 *		inbuff (I)	address of MEMBUFF normally containing NULL
 *				to read from the WordPerfect file,
 *				or containing an input buffer to be processed.
 * Returns:	(int)		most recent character, or EOF for any error.
 * --------------------------------------------------------------------------
 * Notes:     o
 ***************************************************************************/
/* --- entry point --- */
int	wpregetc ( tpfcb, inbuff )
TPFCB	*tpfcb;
MEMBUFF	*inbuff;
{
/* --------------------------------------------------------------------------
Return most recently read character to caller
-------------------------------------------------------------------------- */
int	wpbyte = tpfcb->wpdoc.filebuff->buffchar; /*get current buffer char*/
if ( inbuff != (MEMBUFF *)NULL )	/* user has his own buffer */
  wpbyte = inbuff->buffchar;		/* so use it instead */
end_of_job: return ( wpbyte );		/* most recent char from wpgetc() */
} /* --- end-of-function wpregetc() --- */


/****************************************************************************
 * Function:	wpungetc ( tpfcb, inbuff, wpbyte )
 * Purpose:	Pushes char onto end of unget buffer.
 * --------------------------------------------------------------------------
 * Arguments:	tpfcb (I)	address of file control block
 *				containing file pointer of WP document file.
 *		inbuff (I)	address of MEMBUFF normally containing NULL
 *				to unget the WordPerfect file buffer,
 *				or containing an input buffer to be processed.
 *		wpbyte (I)	int containing char to be pushed on buffer.
 * Returns:	(int)		wpbyte from caller, or EOF for any error.
 * --------------------------------------------------------------------------
 * Notes:     o
 ***************************************************************************/
/* --- entry point --- */
int	wpungetc ( tpfcb, inbuff, wpbyte )
TPFCB	*tpfcb;
MEMBUFF	*inbuff;
int	wpbyte;
{
/* --------------------------------------------------------------------------
Allocations and Declarations
-------------------------------------------------------------------------- */
int	wpunget = EOF;			/* init to signal failure */
unsigned char *buff = tpfcb->wpdoc.filebuff->buffer; /* buffer pointer */
int	buffsize = tpfcb->wpdoc.filebuff->buffsize, /* allocated buffer size */
	bufftail = tpfcb->wpdoc.filebuff->bufftail; /* buffer tail offset */
/* --------------------------------------------------------------------------
Check for user buffer
-------------------------------------------------------------------------- */
if ( inbuff != (MEMBUFF *)NULL )
  {
  int buffhead = inbuff->buffhead;	/* input buffer head offset */
  if ( buffhead <= 0 ) goto end_of_job;	/* already at beginning of buffer */
  (inbuff->buffer)[buffhead-1] = wpbyte; /* insert byte */
  inbuff->buffhead -= 1;		/* bump back to previous byte */
  wpunget = wpbyte;			/* signal success */
  goto end_of_job;
  } /* --- end-of-if(inbuff!=NULL) --- */
/* --------------------------------------------------------------------------
Push character onto unget buffer
-------------------------------------------------------------------------- */
buff[bufftail] = (char)(wpbyte);	/* push char on buffer */
bufftail++;				/* and bump tail */
if ( bufftail >= buffsize )		/* passed end-of-buffer */
  bufftail = 0;				/* so wrap to beginning */
tpfcb->wpdoc.filebuff->bufftail = bufftail; /* reset buffer tail */
wpunget = wpbyte;			/*signal success with caller's byte*/
end_of_job: return ( wpunget );		/* return char from caller or error */
} /* --- end-of-function wpungetc() --- */


/****************************************************************************
 * Function:	wpungets ( tpfcb, inbuff, wpstring )
 * Purpose:	Pushes string onto end of unget buffer.
 * --------------------------------------------------------------------------
 * Arguments:	tpfcb (I)	address of file control block
 *				containing file pointer of WP document file.
 *		inbuff (I)	address of MEMBUFF normally containing NULL
 *				to unget the WordPerfect file buffer,
 *				or containing an input buffer to be processed.
 *		wpstring (I)	address of char string to be pushed on buffer.
 * Returns:	(int)		0 for success, or EOF for any error.
 * --------------------------------------------------------------------------
 * Notes:     o
 ***************************************************************************/
/* --- entry point --- */
int	wpungets ( tpfcb, inbuff, wpstring )
TPFCB	*tpfcb;
MEMBUFF	*inbuff;
char	*wpstring;
{
/* --------------------------------------------------------------------------
Push string onto unget buffer
-------------------------------------------------------------------------- */
int	wpbyte = EOF;
while ( (wpbyte = (int)(*wpstring++))	/* next char from string */
!= 0 )					/* until terminating null */
  if ( (wpbyte = wpungetc(tpfcb,inbuff,wpbyte)) /* push char onto buffer */
  == EOF ) break;			/* quit if error */
return ( wpbyte );			/* return 0, or EOF for error */
} /* --- end-of-function wpungets() --- */


/****************************************************************************
 * Function:	wpread ( tpfcb, inbuff, buffer, itemsz, nitems )
 * Purpose:	Performs an fread(), or takes chars from inbuff.
 * --------------------------------------------------------------------------
 * Arguments:	tpfcb (I)	address of file control block
 *				containing file pointer of WP document file.
 *		inbuff (I)	address of MEMBUFF normally containing NULL
 *				to fread() chars from the WordPerfect file,
 *				or containing an input buffer to be processed.
 *		buffer (O)	address of void returning items read.
 *		itemsz (I)	int containing #bytes/item.
 *		nitems (I)	int containing number of items to be read.
 * Returns:	(int)		nitems, or EOF for any error.
 * --------------------------------------------------------------------------
 * Notes:     o
 ***************************************************************************/
/* --- entry point --- */
int	wpread ( tpfcb, inbuff, buffer, itemsz, nitems )
TPFCB	*tpfcb;
MEMBUFF	*inbuff;
void	*buffer;
int	itemsz;
int	nitems;
{
/* --------------------------------------------------------------------------
Read directly from file
-------------------------------------------------------------------------- */
int	nread = EOF;			/* set #items read for error */
if ( inbuff == (MEMBUFF *)NULL )	/* no inbuff so take from file */
  { nread = fread(buffer,itemsz,nitems,tpfcb->wpdoc.fp);
    goto end_of_job; }
/* --------------------------------------------------------------------------
Read from inbuff
-------------------------------------------------------------------------- */
if ( inbuff != (MEMBUFF *)NULL )	/* user provided input buffer */
  {
  int	nbytes = nitems*itemsz;		/* total #bytes to be read */
  int	buffhead = inbuff->buffhead;	/* buffer head offset */
  int	bufftail = inbuff->bufftail;	/* buffer tail offset */
  if ( buffhead < 0			/* EOF for illegal index or */
  ||   buffhead+nbytes >= bufftail ) goto end_of_job; /* past end-of-buffer */
  memcpy(buffer,(void *)(inbuff->buffer +buffhead),nbytes); /*read from buff*/
  inbuff->buffhead += nbytes;		/* bump past bytes read */
  nread = nitems;			/* set for success */
  goto end_of_job;			/* return byte to caller */
  } /* --- end-of-if(inbuff!=NULL) --- */
end_of_job: return ( nread );		/* #items read, or EOF, to caller */
} /* --- end-of-function wpread() --- */


/****************************************************************************
 * Function:	tpputc ( tpfcb, tpbyte )
 * Purpose:	Writes tpchar to file, or to buffer.
 * --------------------------------------------------------------------------
 * Arguments:	tpfcb (I)	address of file control block
 *				containing file pointer of TexPerfect file.
 *		tpbyte (I)	int containing (ascii) character to be
 *				written.
 * Returns:	(int)		tpbyte written to file (or buffer),
 *				or 0 if ignored, or EOF for any error.
 * --------------------------------------------------------------------------
 * Notes:     o	Call with tpbyte=FLUSH at end-of-job to flush buffer.
 ***************************************************************************/
/* --- entry point --- */
int	tpputc ( tpfcb, tpbyte )
TPFCB	*tpfcb;
int	tpbyte;
{
FILE	*fp = tpfcb->tpfile.fp;		/* file pointer */
int	tpput = EOF;
tpput = fputc(tpbyte,fp);		/* binary write */
return ( tpput );			/* char back to caller */
} /* --- end-of-function tpputc() --- */


/****************************************************************************
 * Function:	tpoutc ( tpfcb, outbuff, tpbyte )
 * Purpose:	Writes tpchar to file, or to buffer.
 * --------------------------------------------------------------------------
 * Arguments:	tpfcb (I)	address of file control block
 *				containing file pointer of TexPerfect file.
 *		outbuff (I)	address of MEMBUFF normally containing NULL
 *				to write to the TeXPerfect file,
 *				or containing an output buffer to be written.
 *		tpbyte (I)	int containing (ascii) character to be
 *				written.
 * Returns:	(int)		tpbyte written to file (or buffer),
 *				or 0 if ignored, or EOF for any error.
 * --------------------------------------------------------------------------
 * Notes:     o
 ***************************************************************************/
/* --- entry point --- */
int	tpoutc ( tpfcb, outbuff, tpbyte )
TPFCB	*tpfcb;
MEMBUFF	*outbuff;
int	tpbyte;
{
/* --------------------------------------------------------------------------
Call tpputc() or write to buffer
-------------------------------------------------------------------------- */
int	tpout = EOF;			/* byte written, init for error */
if ( outbuff == (MEMBUFF *)NULL )	/* no user output buffer */
  tpout = tpputc(tpfcb,tpbyte);		/* pass byte down to tpputc() */
else
  {
  int	bufftail = outbuff->bufftail;	/* buffer tail offset */
  int	buffsize = outbuff->buffsize;	/* buffer size */
  if ( bufftail < 0			/* EOF for illegal index or */
  ||   bufftail >= buffsize ) goto end_of_job; /* past end-of-buffer */
  (outbuff->buffer)[bufftail] = (unsigned char)(tpbyte); /* insert byte */
  outbuff->bufftail += 1;		/* bump to next byte */
  tpout = outbuff->buffchar = tpbyte;	/* store last char in buffer */
  goto end_of_job;			/* return byte to caller */
  } /* --- end-of-if/else(outbuff==NULL) --- */
end_of_job: return ( tpout );		/*char written or EOF back to caller*/
} /* --- end-of-function tpoutc() --- */


/****************************************************************************
 * Function:	tpputs ( tpfcb, outbuff, tpstring, nchars )
 * Purpose:	Writes string to file (or to buffer)
 * --------------------------------------------------------------------------
 * Arguments:	tpfcb (I)	address of file control block
 *				containing file pointer of TexPerfect file.
 *		outbuff (I)	address of MEMBUFF normally containing NULL
 *				to write to the TeXPerfect file,
 *				or containing an output buffer to be written.
 *		tpstring (I)	address of char string to be written.
 *		nchars (I)	int containing #chars to write,
 *				or -1 for string
 * Returns:	(int)		#chars written, or EOF for any error.
 * --------------------------------------------------------------------------
 * Notes:     o
 ***************************************************************************/
/* --- entry point --- */
int	tpputs ( tpfcb, outbuff, tpstring, nchars )
TPFCB	*tpfcb;
MEMBUFF	*outbuff;
unsigned char *tpstring;
int	nchars;
{
/* --------------------------------------------------------------------------
Write string to file or buffer
-------------------------------------------------------------------------- */
int	nwritten = 0;
int	tpbyte = EOF;
if ( nchars <= 0 )			/* string output */
  while ( (tpbyte = (int)(*tpstring++))	/* next char from string */
  != 0 )				/* until terminating null */
    { if ( tpoutc(tpfcb,outbuff,tpbyte) /* write character */
      == EOF ) { nwritten=EOF; break; }	/* quit if error */
      nwritten++; }			/* bump count */
else
  while ( --nchars >= 0 )		/* count #chars wanted */
    { tpbyte = (int)(*tpstring++);	/* next char from buffer */
      if ( tpoutc(tpfcb,outbuff,tpbyte)	/* write character */
      == EOF ) { nwritten=EOF; break; }	/* quit if error */
      nwritten++; }			/* bump count */
return ( nwritten );			/* #chars written, or EOF for error */
} /* --- end-of-function tpputs() --- */


/****************************************************************************
 * Function:	texcommand ( tpfcb, icommand, istring )
 * Purpose:	Returns the istring-th string corresponding to
 *		the TeX (or LaTeX or AMS-TeX) command icommand.
 * --------------------------------------------------------------------------
 * Arguments:	tpfcb (I)	address of file control block
 *				containing TexPerfect control block.
 *		icommand (I)	int containing #defined command mnemonic,
 *				e.g., EJECT.
 *		istring (I)	int containing 0=first(usually only) string,
 *				1=second string, etc.
 * Returns:	(char *)	command string, or NULL for any error.
 * --------------------------------------------------------------------------
 * Notes:     o
 ***************************************************************************/
/* --- entry point --- */
char	*texcommand ( tpfcb, icommand, istring )
TPFCB	*tpfcb;
int	icommand;
int	istring;
{
/* --------------------------------------------------------------------------
Allocations and Declarations
-------------------------------------------------------------------------- */
char	*cmdstr = (char *)NULL;		/* initialize for end-of-command */
/*char	*varlambda();*/			/* replace args in hdr with tpvars */
/*VARDEF  *tpvars = tpfcb->tpvars;*/	/* variable names and values */
int	textype = tpfcb->textype;	/* TEX, LATEX or AMSTEX */
int	itable = 0;			/* texcmds[] struct index */
int	nstrings = (-1);		/* #strings for command, -1=error */
static	int ncalls = 0;			/*initialize arg strings on 1st call*/

/* --------------------------------------------------------------------------
Special initialization of argument strings on first call
-------------------------------------------------------------------------- */
ncalls++;				/* bump call counter */
#if 0
if ( ncalls == 1 )			/* this is first call */
  {
  for ( itable=0; (cmdstr=hdrstrings[itable]) != (char *)NULL; itable++ )
    strcpy(cmdstr,varlambda(tpvars,cmdstr)); /* xlate each header string */
  } /* --- end-of-if(ncalls==1) --- */
#endif

/* --------------------------------------------------------------------------
Find texcmds[] entry with xlation for desired command, and extract string
-------------------------------------------------------------------------- */
/* --- table lookup --- */
for ( itable=0; ;itable++ )		/* test for end-of-table inside */
  {
  if ( (nstrings = texcmds[itable].nstrings) /* #strings in command xlation */
  < 0 ) break;				/* end-of-table trailer, -1=error */
  if ( icommand == texcmds[itable].command ) /* found command */
    if ( textype  == texcmds[itable].textype /* and exact TeX type match */
    ||   ALLTEX   == texcmds[itable].textype ) break; /* or generic type */
  } /* --- end-of-for(itable) --- */
/* --- check results of lookup --- */
if ( nstrings < 0			/* command not found in table */
||   istring >= nstrings )		/* no more strings in command */
  goto end_of_job;			/* null string returned to caller */
/* --- extract requested string and return it --- */
cmdstr = texcmds[itable].strings[(istring<0)?0:istring]; /*string requested*/
if ( istring == SETCMD )		/*caller wants to set command string*/
  strcpy(cmdstr,tpfcb->argstr);		/* string to be set passed in tpfcb */
end_of_job: return ( cmdstr );		/* back to caller with cmd string */
} /* --- end-of-function texcommand() --- */


/****************************************************************************
 * Function:	setmode ( tpfcb, mode )
 * Purpose:	Sets TeX mode to PARMODE, MATHMODE, etc.
 * --------------------------------------------------------------------------
 * Arguments:	tpfcb (I)	address of file control block
 *				containing TexPerfect control block.
 *		mode (I)	int containing mode to be set,
 *				e.g., PARMODE, MATHMODE, etc.
 * Returns:	(int)		mode set after call,
 *				or 0 for any error.
 * --------------------------------------------------------------------------
 * Notes:     o	Call with USERMATH whenever an explicit $ is found in
 *		the input stream.
 ***************************************************************************/
/* --- entry point --- */
int	setmode ( tpfcb, mode )
TPFCB	*tpfcb;
int	mode;
{
/* --------------------------------------------------------------------------
Allocations and Declarations
-------------------------------------------------------------------------- */
int	oldmode = tpfcb->texmode;	/* current mode */
int	newmode = 0;			/* mode upon return (set for error) */
int	commode = (-1);			/* mode for texcommand() display */
static	int isusermode = 0;		/* math mode not explicitly set */
static	int mathindisp = 0;		/* math requested in display math */
/* --- first check whether no mode change required --- */
if ( mode == oldmode )			/* no action required */
  { newmode = oldmode;			/* so just set mode flag */
    goto end_of_job; }			/* and return */
/* --------------------------------------------------------------------------
Mode change probably required
-------------------------------------------------------------------------- */
switch ( oldmode )
  {
  default: goto end_of_job;		/* program bug */
  case PARMODE: /* --- currently in paragraph mode --- */
    switch ( mode )
      {
      default: goto end_of_job;		/* user paragraph mode error */
      case USERMATH: isusermode = 1;	/* set explicit user mode */
      case MATHMODE: newmode = MATHMODE; /* set math mode */
	break;
      case USERDISP: isusermode = 1;	/* set explicit user mode */
      case DISPMATH: newmode = DISPMATH; /* set display math mode */
	break;
      } /* --- end-of-switch(mode) --- */
    break; /* ---end-of-case PARMODE --- */
  case MATHMODE: /* --- currently in math mode --- */
    switch ( mode )
      {
      default: goto end_of_job;		/* user paragraph mode error */
      case USERMATH: isusermode = (1-isusermode); /* flip user mode flag */
      case PARMODE:  newmode = PARMODE;	/* set paragraph mode */
	if ( isusermode )		/* can't flip out of mathmode */
	  { newmode = oldmode;  goto end_of_job; } /*just return same state*/
	break;
      case USERDISP: isusermode = 1;	/* set explicit user mode */
      case DISPMATH: newmode = DISPMATH; /* set display math mode */
	tpputs(tpfcb,NULL,texcommand(tpfcb,PARMODE,0),-1); /*get out of math*/
	break;
      } /* --- end-of-switch(mode) --- */
    break; /* ---end-of-case MATHMODE --- */
  case DISPMATH: /* --- currently in display math mode --- */
    switch ( mode )
      {
      default: goto end_of_job;		/* program bug */
      case USERDISP: isusermode = 1;	/* set explicit user mode */
	    newmode = oldmode;  goto end_of_job; /*just return same state*/
      case USERPAR:  isusermode = 0;	/* reset explicit user mode */
      case PARMODE:  newmode = PARMODE;	/* set paragraph mode */
        commode = PARDISP;		/* but display appropriate command */
	if ( isusermode )		/* can't flip out of mathmode */
	  { newmode = oldmode;  goto end_of_job; } /*just return same state*/
	break;
      case USERMATH:
      case MATHMODE: newmode = oldmode;	/* stay in display math mode */
	mathindisp = (1-mathindisp);	/* flip mathindisp flag */
	tpputs(tpfcb,NULL,texcommand(tpfcb,
		(mathindisp)?MATHMODE:PARMODE,0),-1); /*in/out of math mode*/
	goto end_of_job;
      } /* --- end-of-switch(mode) --- */
    break; /* ---end-of-case DISPMATH --- */
  } /* --- end-of-switch(oldmode) --- */
/* --------------------------------------------------------------------------
Set new mode and return to caller
-------------------------------------------------------------------------- */
tpfcb->texmode = newmode;		/* store new mode */
if ( commode <= 0 ) commode = newmode;	/* default command mode */
tpputs(tpfcb,NULL,texcommand(tpfcb,commode,0),-1); /* write new mode string */
mathindisp = 0;				/* reset flag */
end_of_job: return ( newmode );		/* new mode back to caller */
} /* --- end-of-function setmode() --- */


/****************************************************************************
 * Function:	setwpchartbl ( )
 * Purpose:	Inits wpchartbl and returns its (char ***) address
 * --------------------------------------------------------------------------
 * Arguments:	none
 * Returns:	(char ***)	address of wpchartbl, or NULL for any error.
 * --------------------------------------------------------------------------
 * Notes:     o
 ***************************************************************************/
/* --- entry point --- */
char	***setwpchartbl (  )
{ 
/* --------------------------------------------------------------------------
Allocations and Declarations
-------------------------------------------------------------------------- */
static	int isinit = 0;			/* init table only on 1st call */
static	int maxsets = 12;		/* 12 character set tables */
int	iset = 0;			/* table/set index */
int	ichar = 0;			/* character index */
char	**wpsetptr = (char **)NULL,	/* ptr to set within wpsettbl */
	**wpcharptr = (char **)NULL;	/* ptr to set within wpchartbl */
char	*wpinit = (char *)NULL,		/*character from initialization set*/
	*wpchar = (char *)NULL;		/* addr of char to be initialized */
int	isendofset = 0;			/* true when ENDOFSET init detected */
/* --------------------------------------------------------------------------
Init tables on 1st (and probably only) call
-------------------------------------------------------------------------- */
if ( !isinit )				/* tables not initialized yet */
  {
  isinit = 1;				/* set flag to avoid re-init */
  for ( iset=0; iset<maxsets; iset++ )	/* for each table set */
    {
    wpsetptr = wpsettbl[iset];		/* initialization strings for set */
    wpcharptr = wpchartbl[iset];	/* fully-allocated strings */
    isendofset = 0;			/* wpset endofset not detected yet */
    if ( wpsetptr == NULL ) isendofset=1; /* no xlations for this set */
    for ( ichar=0; ichar<WPTBLSZ; ichar++ ) /* for each char in set */
      {
      wpchar = (char *)(malloc(CHARSZ)); /* allocate memory for char */
      if ( wpchar == (char *)NULL )	/* no more memory */
	return ( NULL );		/* trouble in River City */
      wpcharptr[ichar] = wpchar;	/* store address in table */
      memset((void *)wpchar,0,CHARSZ);	/* initialize char to nulls */
      if ( !isendofset )		/*still have initialization strings*/
	{
	wpinit = wpsetptr[ichar];	/* get ptr to initialization string */
	if ( wpinit == NULL ) continue;	/* undefined char, but not endofset */
	if ( strcmp(wpinit,ENDOFSET) == 0 ) /* found end-of-set delimiter */
	  { isendofset=1; continue; }	/* set flag and stop initialization */
	strcpy(wpchar,wpinit);		/* initialize string */
	} /* --- end-of-if(!isendofset) --- */
      } /* --- end-of-for(ichar) --- */
    } /* --- end-of-for(iset) --- */
  } /* --- end-of-if(!isinit) --- */
return ( wpchartbl );			/*back to caller with table address*/
} /* --- end-of-function setwpchartbl() --- */


/****************************************************************************
 * Function:	tpmalloc ( nbytes )
 * Purpose:	Mallocs a MEMBUFF structure with an nbytes-buffer.
 * --------------------------------------------------------------------------
 * Arguments:	nbytes (I)	int containing #bytes for the buffer,
 *				or <=0 for default MEMBUFFSIZE.
 * Returns:	(MEMBUFF *)	address of allocated MEMBUFF,
 *				or NULL for any error.
 * --------------------------------------------------------------------------
 * Notes:     o
 ***************************************************************************/
/* --- entry point --- */
MEMBUFF	*tpmalloc ( nbytes )
int	nbytes;
{ 
/* --------------------------------------------------------------------------
Malloc and initialize a membuff and its included buffer
-------------------------------------------------------------------------- */
MEMBUFF *membuff = (MEMBUFF *)NULL;	/* membuff to be allocated */
if ( (membuff = (MEMBUFF *)malloc(sizeof(MEMBUFF))) /* allocate membuff */
==   (MEMBUFF *)NULL ) goto end_of_job;	/* failed to malloc, return NULL */
if ( nbytes <= 0 ) nbytes = MEMBUFFSIZE; /* set default buffer size */
if ( (membuff->buffer	= (unsigned char *)(malloc(nbytes))) /*alloc buffer*/
==   (unsigned char *)NULL )		/* failed to malloc buffer */
  { free((void *)membuff); membuff=(MEMBUFF *)NULL; goto end_of_job; }
memset((void *)(membuff->buffer),0,MEMBUFFSIZE); /* init buffer to 0's */
membuff->buffsize = MEMBUFFSIZE;	/* size of malloced buffer */
membuff->buffhead = 0;			/* buffer head */
membuff->bufftail = 0;			/* buffer tail (nothing in buffer) */
membuff->buffchar = 0;			/* trailing function code */
end_of_job: return ( membuff );		/*allocated and initialized membuff*/
} /* --- end-of-function tpmalloc() --- */


/****************************************************************************
 * Function:	tpfree ( membuff )
 * Purpose:	Free a MEMBUFF structure and its included buffer.
 * --------------------------------------------------------------------------
 * Arguments:	membuff (I)	address of membuff struct to be freed.
 * Returns:	(int)		1 if successful, or 0 for any error.
 * --------------------------------------------------------------------------
 * Notes:     o
 ***************************************************************************/
/* --- entry point --- */
int	tpfree ( membuff )
MEMBUFF	*membuff;
{ 
/* --------------------------------------------------------------------------
Free a membuff and its included buffer
-------------------------------------------------------------------------- */
int	status = 0;			/* init retrun status for error */
if ( membuff == (MEMBUFF *)NULL ) goto end_of_job; /* no membuff to free */
if ( membuff->buffer == (unsigned char *)NULL ) /* no buffer to free */
  { free((void *)membuff); goto end_of_job; } /* free membuff, return error */
free((void *)membuff->buffer); free((void *)membuff); status=1; /*ok to free*/
end_of_job: return ( status );		/* 1=success, 0=error */
} /* --- end-of-function tpfree() --- */
/* ----------------------- END-OF-FILE TPTOOLS.C ------------------------- */
