/*************************************************************************
*
*   BM2LTX   -   (c) copyright 1995 claas bontus
*
*   email: bontus@al6000.physik.uni-siegen.de
*
*   You are free to distribute this program and to modify it if you
*   document all changes. You are encouraged to transform BM2LTX to
*   other operating systems.
*
*************************************************************************/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include "pcx_int.h"

/*     GLOBALS   */

static char *use_msg[] =
{ "BM2LTX 1.0   -  (c) copyright 1995  claas bontus \n",
  "usage: BM2LTX infile.pcx outfile.pic\n",
  (unsigned char *)NULL
};


/*     FORWARD  REFERENCES  */

static BOOL pcx_read_init(PCX_WORKBLK *, int, int);
static BOOL pcx_read_extpal(PCX_WORKBLK *);
static BOOL pcx_read_header(PCX_WORKBLK *);
static BOOL pcx_read_line(PCX_WORKBLK *, unsigned char *, int);
static BOOL pcx2ltx(char *,char *);
static BOOL ltx_write(PCX_WORKBLK *,unsigned char *,FILE *,int);

#define min(a,b) (((a)<(b))?(a):(b))


/*
 *************************************************************************
 *
 *  MAIN - Executive Function
 *
 *  Setup: int main
 *         ( int argc,
 *           char **argv
 *         )
 *
 *************************************************************************
*/

int main(int argc, char **argv)
{ int i;
  BOOL status = FALSE;
  if(argc == 3)
    status = pcx2ltx(argv[1],argv[2]);
  if(status == FALSE)  /* PRINT COPYRIGHT */
  { i=0;
    while(use_msg[i] != NULL)
      fputs(use_msg[i++], stderr);
  } /* endif */
  return 0;
} /* main */


/*
 *************************************************************************
 *
 *  pcx2ltx - Read PCX-file write LaTeX-file
 *
 *  Setup: static BOOL pcx2ltx
 *         ( char *ReadName,
 *           char *WriteName
 *         )
 *
 *************************************************************************
*/

static BOOL pcx2ltx( char *ReadName, char *WriteName)
{ int bpline;
  int line_num;
  int max_lines;
  unsigned char *linep;
  FILE *OutFile;
  BOOL status = TRUE;
  PCX_WORKBLK *wbp;

  /*  Open input file  */

  if((wbp = pcx_open(ReadName,FALSE)) == (PCX_WORKBLK *)NULL)
  { fputs("File open for reading error\n",stderr);
    return FALSE;
  }

  /*  Open output file */

  if((OutFile=fopen(WriteName,"wb"))==NULL)
  { fputs("File open for writing error\n",stderr);
    (void) pcx_close(wbp);
    return FALSE;
  } /* endif */

  /*  Initialisation  */

  if(pcx_read_init(wbp,13,0)==FALSE)
  { fputs("Initialisation failure\n",stderr);
    (void) pcx_close(wbp);
    return FALSE;
  } /* endif */

  /*  This version of BM2LTX handles only PCX-files with one color plane */

  if(wbp->header.nplanes != 1)
  { fputs("BM2LTX cannot process PCX-files with multiple color planes\n",
          stderr);
    (void) pcx_close(wbp);
    fclose(OutFile);
    return FALSE;
  }

  /*  Calculate line number of picture  */

  max_lines = wbp->header.ylr - wbp->header.yul + 1;

  /*  and number of bytes per line  */

  bpline = wbp->header.bppscan * wbp->header.nplanes;

  /*  Write file header  */

  fprintf( OutFile,"%%******************************************************\r\n");
  fprintf( OutFile,"%%  %s\r\n",WriteName);
  fprintf( OutFile,"%%  Created from %s  with BM2LTX\r\n",ReadName);
  fprintf( OutFile,"%%******************************************************\r\n\r\n");
  fprintf( OutFile,"\\begin{picture}(%d,%d)\r\n",
           wbp->header.xlr - wbp->header.xul +1,
           max_lines
         );

  /*  Print picture resolution  */

  printf("Resolution: %d x %d",wbp->header.xlr - wbp->header.xul +1,
                               wbp->header.ylr - wbp->header.yul +1
        );
  if((linep = (unsigned char *) malloc(bpline)) == (unsigned char *) NULL)
  { fputs("malloc returned NULL\n",stderr);
    status = FALSE;
  }
  else
  { if(status)

      /*  Seek beginning of data  */

      if(fseek(wbp->fp, (long) (sizeof(PCX_HDR)) , SEEK_SET) != 0)
      { fputs("fseek error\n",stderr);
        status=FALSE;
      }
    if(status)
    { 
      /*  Read data line by line  */

      for (line_num=0; line_num < max_lines; ++line_num)
      { if((status=pcx_read_line(wbp,linep,bpline))==FALSE)
        { fputs("pcx_read_line error\n",stderr);
          break;
        }

        /*  and write data to output file  */

        ltx_write(wbp, linep, OutFile,max_lines-line_num);
      } /* for */
    } /* endif */
  free(linep);
  } /* endif */

  /*  Write last line to output file  */

  if(status)
    fprintf( OutFile,"\\end{picture}\r\n");
  if(pcx_close(wbp) == FALSE)
  { fputs("pcx_close error\n",stderr);
    status = FALSE;
  }
  if(fclose(OutFile)==EOF)
  { fputs("fclose error\n",stderr);
    status = FALSE;
  }
  return status;
} /* pcx2ltx */


/*
 *************************************************************************
 *
 *  ltx_write - Write one line to output file
 *
 *  Setup:  static BOOL ltx_write
 *          ( PCX_WORKBLK *wbp,
 *            unsigned char *linep,
 *            FILE *OutFile,
 *            int line_num 
 *          )
 *
 *************************************************************************
*/

static BOOL ltx_write
            ( PCX_WORKBLK *wbp,
              unsigned char *linep,
              FILE *OutFile,
              int line_num
            )
{ BYTE bpp=wbp->header.bppixel;
  WORD Pixel=wbp->header.xlr - wbp->header.xul +1;
  BYTE MaskArray[]={0,128,192,224,240,248,252,254,255};
  BYTE CurrentMask=MaskArray[bpp];
  BYTE CurrentPixel;
  int  i,i1=0;
  int  StartNonZero=-1;
  int  CountNonZero=0;

  /*  Cut off overstanding bits  */

  linep[wbp->num_bytes -1 ] |= (( !wbp->mask)&255 );

  /*  Process line pixel by pixel  */

  for(i=0;i<Pixel;++i)
  { CurrentPixel=linep[i1]&CurrentMask;
    if((CurrentMask=CurrentMask>>bpp)==0)
    { CurrentMask=MaskArray[bpp];
      ++i1;
    }

    /*  Count number of succeeding blacks */

    if(!CurrentPixel)
    { if(StartNonZero==-1)StartNonZero=i;
      ++CountNonZero;
    } /* if */
    else

    /*  Check if this is first white after a number of blacks  */

    { if(CountNonZero)
      { fprintf( OutFile,"\\put(%d,%d){\\rule{%dpt}{1pt}}\r\n",
                 StartNonZero,line_num,CountNonZero
               );
        CountNonZero=0;
        StartNonZero=-1;
      }
    } /* endif */
  } /* for */

  /*  If last pixel of line is black  */

  if(CountNonZero)
    fprintf( OutFile,"\\put(%d,%d){\\rule{%dpt}{1pt}}\r\n",
             StartNonZero,line_num,CountNonZero
           );
  return TRUE;
} /* ltx_write */


/*
 *************************************************************************
 *
 *  PCX_READ_INIT - Initialize PCX Workblock For Reading
 *
 *  Purpose:    To initialize a PCX image file workblock for reading.
 *
 *  Setup:      static BOOL pcx_read_init
 *              (
 *                PCX_WORKBLK *wbp,
 *                int vmode,
 *                int page
 *              )
 *
 *  Where:      wbp is a PCX workblock pointer.
 *              vmode is the MS-DOS video mode.  Valid values are:
 *
 *                PCX_HERC -    720 x 348 Hercules monochrome
 *                0x04 -        320 x 200 4-color CGA
 *                0x05 -        320 x 200 4-color CGA (color burst off)
 *                0x06 -        640 x 200 2-color CGA
 *                0x0d -        320 x 200 16-color EGA/VGA
 *                0x0e -        640 x 200 16-color EGA/VGA
 *                0x0f -        640 x 350 2-color EGA/VGA
 *                0x10 -        640 x 350 16-color EGA/VGA
 *                0x11 -        640 x 480 2-color VGA
 *                0x12 -        640 x 480 16-color VGA
 *                0x13 -        320 x 200 256-color VGA
 *
 *              page is the video display page number.  Valid values are:
 *
 *                Mode PCX_HERC - 0 or 1
 *                Mode 0x0d     - 0 to 7
 *                Mode 0x0e     - 0 to 3
 *                Mode 0x0f     - 0 or 1
 *                Mode 0x10     - 0 or 1
 *                All Other      - 0
 *
 *  Return:     TRUE if successful; otherwise FALSE.
 *
 *************************************************************************
 */

static BOOL pcx_read_init
(
  PCX_WORKBLK *wbp,
  int vmode,
  int page
)
{
  int width;                    /* Display width                        */
  int leftover;                 /* Number of unseen bits                */
  BOOL status = TRUE;           /* Return status                        */

  /* Read the file header                                               */

  if ((pcx_read_header(wbp)) == FALSE)
    return (FALSE);

  /* Initialize the workblock color palette pointer                     */

  wbp->palettep = wbp->header.palette;
  wbp->epal_flag = FALSE;

  /* Read the extended palette (if any)                                 */

  if (wbp->header.version == 5)
    if (pcx_read_extpal(wbp) == FALSE)
      return (FALSE);

  /* Initialize the display page address offset                         */

  wbp->page_offset = (unsigned long) 0L;

  switch (vmode)        /* Select PCX line display function             */
  {
    case PCX_HERC:      /* 720 x 348 Hercules monochrome                */

      /* Hercules monochrome display adapter supports 2 pages           */

      wbp->page_offset = 0x08000000L * (unsigned long) page;

      /* Calculate display width in pixels                              */

      width = min((wbp->header.xlr - wbp->header.xul + 1), 720);

      /* Calculate number of bytes to display                           */

      wbp->num_bytes = (width + 7) >> 3;

      /* Calculate mask for leftover bits                               */

      if ((leftover = width & 7) != 0)
        wbp->mask = (0xff << (8 - leftover)) & 0xff;
      else
        wbp->mask = 0xff;

/*      wbp->pcx_funcp = pcx_put_herc;    /* Set the display function ptr */

      break;

    case 0x04:          /* 320 x 200 4-color CGA                        */
    case 0x05:          /* 320 x 200 4-color CGA (color burst off)      */

      /* Calculate display width in pixels                              */

      width = min((wbp->header.xlr - wbp->header.xul + 1), 320);

      /* Calculate number of bytes to display                           */

      wbp->num_bytes = (width + 3) >> 2;

      /* Calculate mask for leftover bits                               */

      if ((leftover = (width & 3) << 1) != 0)
        wbp->mask = (0xff << (8 - leftover)) & 0xff;
      else
        wbp->mask = 0xff;

/*      wbp->pcx_funcp = pcx_put_cga;     /* Set the display function ptr */

      break;

    case 0x06:          /* 640 x 200 2-color CGA                        */

      /* Calculate display width in pixels                              */

      width = min((wbp->header.xlr - wbp->header.xul + 1), 640);

      /* Calculate number of bytes to display                           */

      wbp->num_bytes = (width + 7) >> 3;

      /* Calculate mask for leftover bits                               */

      if ((leftover = width & 7) != 0)
        wbp->mask = (0xff << (8 - leftover)) & 0xff;
      else
        wbp->mask = 0xff;

/*      wbp->pcx_funcp = pcx_put_cga;     /* Set the display function ptr */

      break;

    case 0x0d:          /* 320 x 200 16-color EGA/VGA                   */
    case 0x0e:          /* 640 x 200 16-color EGA/VGA                   */
    case 0x0f:          /* 640 x 350 2-color EGA/VGA                    */
    case 0x10:          /* 640 x 350 16-color EGA/VGA                   */
    case 0x11:          /* 640 x 480 2-color VGA                        */
    case 0x12:          /* 640 x 480 16-color VGA                       */

      switch (vmode)    /* Initialize the display adapter page offset   */
      {
        case 0x0d:      /* 320 x 200 16-color EGA/VGA (8 pages maximum) */

          wbp->page_offset = 0x02000000L * (unsigned long) page;

          break;

        case 0x0e:      /* 640 x 200 16-color EGA/VGA (4 pages maximum) */

          wbp->page_offset = 0x04000000L * (unsigned long) page;

          break;

        case 0x0f:      /* 640 x 350 2-color EGA/VGA (2 pages maximum)  */
        case 0x10:      /* 640 x 350 16-color EGA/VGA (2 pages maximum) */

          wbp->page_offset = 0x08000000L * (unsigned long) page;

          break;

        default:        /* All other modes support only one page        */

          break;
      }

      /* Calculate display width in pixels                              */

      width = min((wbp->header.xlr - wbp->header.xul + 1), 640);

      /* Calculate number of bytes to display                           */

      wbp->num_bytes = (width + 7) >> 3;

      /* Calculate mask for leftover bits                               */

      if ((leftover = width & 7) != 0)
        wbp->mask = (0xff << (8 - leftover)) & 0xff;
      else
        wbp->mask = 0xff;

/*      wbp->pcx_funcp = pcx_put_ega;     /* Set the display function ptr */

      break;

    case 0x13:          /* 320 x 200 256-color VGA                      */

      /* Calculate number of bytes to display                           */

      wbp->num_bytes = min((wbp->header.xlr - wbp->header.xul + 1), 320);

      wbp->mask = 0;  /* Dummy parameter                                */

/*      wbp->pcx_funcp = pcx_put_vga;     /* Set the display function ptr */

      break;

    default:            /* Other display adapters not supported         */

      status = FALSE;

      break;
  }

  return (status);
}

/*
 *************************************************************************
 *
 *  PCX_READ_HEADER - Read PCX File Header
 *
 *  Purpose:    To read and validate a PCX file header.
 *
 *  Setup:      static BOOL pcx_read_header
 *              (
 *                PCX_WORKBLK *wbp
 *              )
 *
 *  Where:      wbp is a PCX image file workblock pointer.
 *
 *  Return:     TRUE if successful; otherwise FALSE.
 *
 *  Result:     The file header is read into the "header" member of the
 *              PCX workblock.
 *
 *************************************************************************
 */

static BOOL pcx_read_header
(
  PCX_WORKBLK *wbp
)
{
  BOOL status = TRUE;   /* Status flag                                  */
  PCX_HDR *hdrp;        /* PCX file header buffer pointer               */

  hdrp = &(wbp->header);        /* Initialize the file header pointer   */

  /* Read the file header                                               */

  if (fseek(wbp->fp, 0L, SEEK_SET) != 0)
    status = FALSE;

  if (status == TRUE)
    if (fread(hdrp, sizeof(PCX_HDR), 1, wbp->fp) != 1)
      status = FALSE;

  /* Validate the PCX file format                                       */

  if (status == TRUE)
    if ((hdrp->pcx_id != 0x0a) || (hdrp->encoding != 1))
      status = FALSE;

  return (status);
}

/*
 *************************************************************************
 *
 *  PCX_READ_EXTPAL - Read Extended Palette
 *
 *  Purpose:    To read an extended (256-color) palette (if it exists).
 *
 *  Setup:      static BOOL pcx_read_extpal
 *              (
 *                PCX_WORKBLK *wbp
 *              )
 *
 *  Where:      wbp is a PCX image file workblock pointer.
 *
 *  Return:     TRUE if successful; otherwise FALSE.
 *
 *  Note:       It is possible for a PCX image file without an appended
 *              256-color palette to have the value 0x0c as the 769th byte
 *              (the location of the extended palette indicator byte) from
 *              the end of the file (i.e. - in the encoded image data
 *              section).  This function will misinterpret the following
 *              768 bytes of encoded image data as an extended palette.
 *
 *              This problem will only occur if an attempt is made to
 *              display a PCX image using the wrong MS-DOS video mode.  It
 *              can be detected by decoding the image data and using
 *              "ftell" to note the file position of the end of the
 *              encoded image data section, then comparing it to the file
 *              position of the indicator byte.  If the supposed indicator
 *              byte is located within the encoded image data section, the
 *              indicator byte is invalid and so the file header palette
 *              should be used instead.
 *
 *************************************************************************
 */

static BOOL pcx_read_extpal
(
  PCX_WORKBLK *wbp
)
{
  int indicator;        /* PCX extended palette indicator               */

  /* Position the file pointer to the extended palette indicator byte   */

  if (fseek(wbp->fp, -769L, SEEK_END) != 0)
    return (FALSE);

  /* Read the (assumed) extended palette indicator byte                 */

  if ((indicator = getc(wbp->fp)) == EOF)
    return (FALSE);

  if (indicator == PCX_EPAL_FLAG)       /* Check for indicator byte     */
  {
    /* Allocate an extended palette buffer                              */

    if ((wbp->palettep = (PCX_PAL *) calloc(sizeof(PCX_PAL),
        PCX_EPAL_SIZE)) == (PCX_PAL *) NULL)
      return (FALSE);

    /* Read the extended palette                                        */

    if (fread(wbp->palettep, sizeof(PCX_PAL), PCX_EPAL_SIZE, wbp->fp) !=
        PCX_EPAL_SIZE)
    {
      free(wbp->palettep);      /* Free the extended palette buffer     */
      return (FALSE);
    }

    wbp->epal_flag = TRUE;      /* Indicate extended palette present    */
  }

  return (TRUE);
}

/*
 *************************************************************************
 *
 *  PCX_READ_LINE - Read PCX Line
 *
 *  Purpose:    To read an encoded line (all color planes) from a PCX-
 *              format image file and write the decoded data to a line
 *              buffer.
 *
 *  Setup:      static BOOL pcx_read_line
 *              (
 *                PCX_WORKBLK *wbp,
 *                unsigned char *linep,
 *                int bpline
 *              )
 *
 *  Where:      wbp is a PCX image file workblock pointer.
 *              linep is a PCX scan line buffer pointer.
 *              bpline is the number of bytes per scan line (all color
 *                planes).
 *
 *  Return:     TRUE if successful; otherwise FALSE.
 *
 *************************************************************************
 */

static BOOL pcx_read_line
(
  PCX_WORKBLK *wbp,
  unsigned char *linep,
  int bpline
)
{
  int data;             /* Image data byte                              */
  int count;            /* Image data byte repeat count                 */
  int offset = 0;       /* Scan line buffer offset                      */

  while (offset < bpline)       /* Decode current scan line             */
  {
    if ((data = getc(wbp->fp)) == EOF)  /* Get next byte                */
      return (FALSE);

    /* If top two bits of byte are set, lower six bits show how         */
    /* many times to duplicate next byte                                */

    if ((data & PCX_COMP_FLAG) == PCX_COMP_FLAG)
    {
      count = data & PCX_COMP_MASK;     /* Mask off repeat count        */

      if ((data = getc(wbp->fp)) == EOF)        /* Get next byte        */
        return (FALSE);

      memset(linep, data, count);       /* Duplicate byte               */
      linep += count;
      offset += count;
    }
    else
    {
      *linep++ = (unsigned char) data;  /* Copy byte                    */
      offset++;
    }
  }

  return (TRUE);
}

