/*

  t1extremes
  ==========

  Scan the outlines of a Type 1 font in order to see if some extremes
  are missing. Type 1 specifications strongly suggest to add a point in
  the outline path at such extreme places.

  Copyright (c) 2003 Thomas Baruchel
  
  Permission is hereby granted, free of charge, to any person obtaining
  a copy of this software and associated documentation files (the
  "Software"), to deal in the Software without restriction, including
  without limitation the rights to use, copy, modify, merge, publish,
  distribute, sublicense, and/or sell copies of the Software, and to
  permit persons to whom the Software is furnished to do so, subject to
  the following conditions:
  
  The above copyright notice and this permission notice shall be
  included in all copies or substantial portions of the Software.
  
  THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
  EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
  MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.
  IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY
  CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT,
  TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE
     SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
*/

#include <stdio.h>
#include <stdlib.h>
#include <t1lib.h>
#include <math.h>

int usage(void) {
  fprintf(stderr,
    "Usage: t1extremes [-v] file\n    -v  verbose output\n");
  return(EXIT_FAILURE);
}

void position(double t, int x0, int y0, int x1, int y1,
                        int x2, int y2, int x3, int y3, char *name,
                        unsigned char d, unsigned char v) {
  auto double x =
    (double) x0 + ((double) (3*(-x0+x1)))*t
    + ((double) (3*(x0-2*x1+x2)))*t*t
    + ((double) (-x0+3*x1-3*x2+x3))*t*t*t;
  auto double y =
    (double) y0 + ((double) (3*(-y0+y1)))*t
    + ((double) (3*(y0-2*y1+y2)))*t*t
    + ((double) (-y0+3*y1-3*y2+y3))*t*t*t;
  fprintf(stdout,"/%s : missing an extreme at (%.3f , %.3f) with a",name,x,y);
  if(d==0) fprintf(stdout," vertical tangent\n");
    else fprintf(stdout,"n horizontal tangent\n");
  if(v!=0) {
    auto double d1 = (x0-x)*(x0-x) + (y0-y)*(y0-y);
    auto double d2 = (x3-x)*(x3-x) + (y3-y)*(y3-y);
    fprintf(stdout,"    closest point of the outline is (");
    if(d1<d2) fprintf(stdout,"%d , %d), distance = %.3f\n",x0,y0,sqrt(d1));
    else fprintf(stdout,"%d , %d), distance = %.3f\n",x3,y3,sqrt(d2));
  }
}

void missing_extremes(T1_OUTLINE *outline, char *name, unsigned char v) {
  auto int xpos = 0;
  auto int ypos = 0;
  do {
    switch(((T1_PATHSEGMENT *) outline)->type) {
      case T1_PATHTYPE_MOVE:
      case T1_PATHTYPE_LINE: {
        xpos += (int) T1_NEARESTPOINT(((T1_PATHSEGMENT *) outline)->dest.x);
        ypos -= (int) T1_NEARESTPOINT(((T1_PATHSEGMENT *) outline)->dest.y);
        break;
      }
      case T1_PATHTYPE_BEZIER: {
        auto int x0 = xpos;
        auto int y0 = ypos;
        auto int x1 = 
          xpos+T1_NEARESTPOINT(((T1_BEZIERSEGMENT *) outline)->B.x);
        auto int y1 = 
          ypos-T1_NEARESTPOINT(((T1_BEZIERSEGMENT *) outline)->B.y);
        auto int x2 = 
          xpos+T1_NEARESTPOINT(((T1_BEZIERSEGMENT *) outline)->C.x);
        auto int y2 = 
          ypos-T1_NEARESTPOINT(((T1_BEZIERSEGMENT *) outline)->C.y);
        auto int a,b,c,delta;
        xpos += (int) T1_NEARESTPOINT(outline->dest.x);
        ypos -= (int) T1_NEARESTPOINT(outline->dest.y);
        a = 3*(-x0+3*x1-3*x2+xpos);
        b = 6*(x0-2*x1+x2);
        c = 3*(-x0+x1);
        delta = b*b-4*a*c;
        if(delta>=0) {
          auto double t1 = (-b-sqrt(delta))/(2*a);
          auto double t2 = (-b+sqrt(delta))/(2*a);
          if((t1>0)&&(t1<1)) position(t1,x0,y0,x1,y1,x2,y2,xpos,ypos,name,0,v);
          if((t2>0)&&(t2<1)) position(t2,x0,y0,x1,y1,x2,y2,xpos,ypos,name,0,v);
        }
        a = 3*(-y0+3*y1-3*y2+ypos);
        b = 6*(y0-2*y1+y2);
        c = 3*(-y0+y1);
        delta = b*b-4*a*c;
        if(delta>=0) {
          auto double t1 = (-b-sqrt(delta))/(2*a);
          auto double t2 = (-b+sqrt(delta))/(2*a);
          if((t1>0)&&(t1<1)) position(t1,x0,y0,x1,y1,x2,y2,xpos,ypos,name,1,v);
          if((t2>0)&&(t2<1)) position(t2,x0,y0,x1,y1,x2,y2,xpos,ypos,name,1,v);
        }
        break;
      }
    }
  } while ((outline = outline->link) != NULL);
}

int main(int argc, char **argv)
{
  static signed int i;
  static unsigned char j=0;
  static char *fontname = NULL;
  static unsigned char v = 0;

  for(i=1;i<argc;i++) {
    if((argv[i][0]=='-')
        && (argv[i][1]=='v')
        && (argv[i][2]=='\0')) v = 1;
    else if(fontname==NULL) fontname = argv[i];
      else exit(usage());
  }
  if(fontname==NULL) exit(usage());
  if (T1_InitLib(NO_LOGFILE)==NULL) {
    fprintf(stderr, "Initialization of t1lib failed\n");
    exit(EXIT_FAILURE);
  }
  i = T1_AddFont(fontname);
  if(i<0) {
    T1_CloseLib();
    if(i==-1) {
      fprintf(stderr, "Font file could not be located\n");
      exit(EXIT_FAILURE);
    }
    fprintf(stderr, "Memory allocation failure\n");
    exit(EXIT_FAILURE);
  }
  if(T1_LoadFont(i)==-1) {
    T1_CloseLib();
    fprintf(stderr, "t1lib could not load the font\n");
    exit(EXIT_FAILURE);
  }
  do {
    auto T1_OUTLINE *myoutline;
    missing_extremes(myoutline = T1_GetCharOutline(i,j,1000,NULL),
                     T1_GetCharName(i,j),v);
    T1_FreeOutline(myoutline);
  } while(j++<255);
  T1_CloseLib();
  return(EXIT_SUCCESS);
}
